/** @file   sprite.h
 * @brief   Declaraction and implementation of Sprite - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_SPRITE_H
#define H_ENG2D_SPRITE_H

#include "eng2d_dll.h"
#include "vec2d.h"

namespace eng2d {


/** @class  Sprite
 * @brief   Simple wrapper for using allegro's bitmaps as sprites.
 * @author  Tomi Lamminsaari
 *
 * 
 */
class DLLIMPORT Sprite
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================



  ///
  /// Public members
  /// ==============
  
  /** Position. */
  Vec2D   m_pos;
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	inline Sprite() :
    m_pBitmap( 0 ),
    m_type( 0 ),
    m_pos( 0,0 )
  {
  }


  /** Constructs new Sprite and attaches the given bitmap to it.
   * @param     pB                Pointer to bitmap containing our graphics
   */
  inline Sprite( BITMAP* pB ) :
    m_pBitmap( pB ),
    m_type( 0 ),
    m_pos( 0,0 )
  {
  }
  
  
  /** Constructs new Sprite and attaches the given rle-sprite to it.
   * @param     pS                Pointer to rle-sprite containing our graphics
   */
  inline Sprite( RLE_SPRITE* pS ) :
    m_pBitmap( pS ),
    m_type( 1 ),
    m_pos( 0,0 )
  {
  }
  

	/** Destructor
   */
	inline virtual ~Sprite() { }


	/** Copy constructor.
   * @param     rO                Reference to another Sprite
   */
  inline Sprite( const Sprite& rO ) :
    m_pBitmap( rO.m_pBitmap ),
    m_type( rO.m_type ),
    m_pos( rO.m_pos )
  {
  }
  

	/** Assignment operator
   * @param     rO                Reference to another Sprite
   * @return    Reference to us.
   */
  inline Sprite& operator = ( const Sprite& rO )
  {
    if ( this != &rO ) {
      m_pBitmap = rO.m_pBitmap;
      m_type = rO.m_type;
      m_pos = rO.m_pos;
    }
    return *this;
  }



  ///
  /// Methods
  /// =======

  /** Attaches new bitmap pointer
   * @param     pB                Pointer to new bitmap data
   */
  inline void attach( BITMAP* pB )
  {
    m_type = 0;
    m_pBitmap = pB;
  }

  /** Attaches the given rle-sprite to this object
   * @param     pS                Pointer to rle-sprite
   */
  inline void attach( RLE_SPRITE* pS )
  {
    m_type = 1;
    m_pBitmap = pS;
  }
  

  /** Sets the position
   * @param     pos               New position
   */
  inline void position( const Vec2D& pos )
  {
    m_pos = pos;
  }
  

  /** Redraws the sprite on given bitmap
   * @param     pB                Pointer to bitmap we draw the sprite
   */
  inline void draw( BITMAP* pB )
  {
    if ( m_type == 0 ) {
      draw_sprite( pB, reinterpret_cast<BITMAP*>(m_pBitmap),
                   static_cast<int>( m_pos.x() ),
                   static_cast<int>( m_pos.y() ) );
    } else if ( m_type == 1 ) {
      draw_rle_sprite( pB, reinterpret_cast<RLE_SPRITE*>( m_pBitmap ),
                       static_cast<int>( m_pos.x() ),
                       static_cast<int>( m_pos.y() ) );
    }
  }


  ///
  /// Getter methods
  /// ==============
  
  /** Returns the pointer to the bitmap.
   * @return    Pointer to the bitmap or null-pointer if we have rle-sprite.
   */
  inline BITMAP* al_BITMAP() const
  {
    if ( m_type == 1 ) {
      return 0;
    }
    return reinterpret_cast<BITMAP*>( m_pBitmap );
  }
  
  
  
  /** Returns the pointer to the rle-sprite.
   * @return    Pointer to rle-sprite or nullpointer if we have normal bitmap
   */
  inline RLE_SPRITE* al_RLESPRITE() const
  {
    if ( m_type == 0 ) {
      return 0;
    }
    return reinterpret_cast<RLE_SPRITE*>( m_pBitmap );
  }
  
  /** Returns the position
   * @return    Position
   */
  inline Vec2D position() const
  {
    return m_pos;
  }


protected:

  ///
  /// Members
  /// =======

  /** Pointer to bitmap data */
  void*   m_pBitmap;
  
  /** Type of this bitmap */
  int     m_type;
  

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
